#ifndef AHLGREN_EMULATED_NRSAMPLE
#define AHLGREN_EMULATED_NRSAMPLE

#include "program.h"
#include "fitness.h"
#include "bottom.h"

namespace lp {
	stat Program::generalize_emulated_nrsample(const sign_t& sign)
	{
		// Cache parameters
		const int param_noise = params.force_int(parameters::noise);
		const int param_csize = params.force_int(parameters::csize);
		//const bool param_memo = params.is_set(parameters::memoize);
		const bool param_prune_consistent = params.is_set(parameters::prune_consistent);
		const bool param_prune_inconsistent = params.is_set(parameters::prune_inconsistent);
		const bool param_enforce_io_spec = params.is_set(parameters::enforce_io_spec);

		//std::cerr << "Generalize_nrsample\n";
		// Pick fitness function
		auto fitness = pick_fitness_function<bsf_type>(params);
		if (!fitness) {
			// Set default: lex
			fitness.reset(new fitness_lex<bsf_type>(params.force_int(parameters::terminate)));
		}

		// Bitstring
		bitstring bs;
		std::set<bitstring> memo; // memoization
		std::set<bitstring> pruned_down; // all candidates below these ones are ignored
		std::set<bitstring> pruned_up; // all candidates above this one are ignored
		io_map iomap;

		auto init_search = [&]
		(Program& thy, Functor& bottom, int bsize, const vector<Mode>& modes, 
			list<clause>& pex, list<Program::clause>& nex, Constraints&, stat& sts) -> bsf_type
		{
		    //std::cerr << "Calling init...\n";
			bs.clear();
			memo.clear();
			pruned_down.clear();
			pruned_up.clear();
			fitness->init(bottom,modes,thy,pex,nex,sts[stat::exc]);
			bs.resize(bsize,0);
			if (param_enforce_io_spec) iomap = make_io_map(bottom,modes);
			return bsf_type();
		};

		auto find_candidate = [&]
		(Program& thy, list<clause>& pex, list<Program::clause>& nex, Functor& bottom, int bsize, const vector<Mode>& modes, const bsf_type& bsf, 
			Constraints&, stat& sts, deadline_t deadline) -> bsf_type
		{
			// Search sequence: 000 -> 001 -> 010 -> 011 -> 100 -> 101 -> 110 -> 111

			// Iterate until we find a valid solution
			decltype(pruned_down.begin()) at;
			bool finished = false;
			bsf_type sol;
			const int MAX_ITER = std::numeric_limits<int>::max(); //20000; // TODO: param
			int k = 0;
			for ( ; k < MAX_ITER && !finished; ++k) {
				// Check Time Out
				if (std::chrono::steady_clock::now() > deadline) {
					throw time_out();
				}

				//if (param_memo && memo.find(bs) != memo.end()) {
				//	// Revisit
				//	DEBUG_INFO(cout << "Skipping revisit: " << bs << "\n");
				//} else
				if (param_prune_consistent &&
					(at = std::find_if(pruned_down.begin(),pruned_down.end(),[&](const bitstring& g){ return is_subset(g,bs); })) != pruned_down.end() ) {
						// pruned away
						DEBUG_INFO(cout << "Skipping pruned_down: " << bs << " from " << *at << "\n");
				} else if (param_prune_inconsistent &&
					(at = std::find_if(pruned_up.begin(),pruned_up.end(),[&](const bitstring& g){ return is_subset(bs,g); })) != pruned_up.end() ) {
						// pruned away
						DEBUG_INFO(cout << "Skipping pruned_up: " << bs << " from " << *at << "\n");
				} else {
					// Make candidate from bottom clause and mask
					sol = bs;
					mask_bottom(bottom,sol.mask());

					if (param_enforce_io_spec && !is_mode_conformant(bottom,thy.params,iomap)) {
						// Solution violates modes, tag as invalid
						sol.set_invalid();
						++sts[stat::invalid];
						finished = true;
					} else {
						DEBUG_INFO( cout << "Generated Solution: " << sol << "\n" );
						++sts[stat::fitc];
						fitness->evaluate(bottom,sol,thy,pex,nex,bsf,sts[stat::exc],deadline);
						if (sol.is_consistent(param_noise)) {
							++sts[stat::conc];
							if (param_prune_consistent) {
								pruned_down.insert(sol.mask());
							}
						} else {
							++sts[stat::inconc];
							if (param_prune_inconsistent) {
								pruned_up.insert(sol.mask());
							}
						}
						//if (param_memo) {
						//	memo.insert(sol.mask());
						//}
						finished = true;
					}
				}
				// Regardless of valid or not...
				// Make next bs
				for (;;) { // until csize satisfied
					// Check Time Out
					if (std::chrono::steady_clock::now() > deadline) {
						throw time_out();
					}
					auto b = bs.rbegin();
					for ( ; b != bs.rend(); ++b) {
						if (*b == 0) {
							// Flip this to 1 and we're done
							*b = 1;
							break;
						} else {
							assert(*b == 1);
							// Flip this to 0
							*b = 0;
						}
					}
					if (b == bs.rend()) {
						// No more solutions, this is the last
						if (finished && std::count(bs.begin(),bs.end(),1) <= param_csize) throw sol;
						else throw search_completed();
					}
					if (std::count(bs.begin(),bs.end(),1) <= param_csize) {
						break;
					} // else: continue
				} // until we get the next bs satisfying csize
			}

			if (k >= MAX_ITER) {
				std::cerr << "WARNING: could not produce mode conformant solution after " << MAX_ITER << " iterations\n";
				throw search_aborted();
			}
			return sol;
		};

		return this->generalize(sign,init_search,find_candidate);
	}


}



#endif


